% Luca Guglielmetti, Zürich 2009
% v. 0.6
% compute calculate backwater curve (longitudinal profile of the liquid's surface)
% solving the energy balance equation and assuming a steady flow 

clear;

% load profile and section data
data = load('dtm_25_cal.txt');

l = flipud(data(:,1));       % length
z = flipud(data(:,2));       % elevation

% cross section parameters
b1 = flipud(data(:,3));
b2 = flipud(data(:,4));
h1 = flipud(data(:,5));
b3 = flipud(data(:,6));
h2 = flipud(data(:,7));

n_manning = flipud(data(:,8));
changepoint = flipud(data(:,9));

% constants
N_ = length(l);               % # sections
I_ = 20;                      % max iterations
ERROR_TOLERANCE_ = 0.1;       % max error in the guess [m]

% variables
n = 0;                        % current cross section iteration
i = 0;                        % current h-iteration

% temp values needed by the algorithm 
h = zeros(N_,I_);             % estimated wassersaeule
h_c = zeros(N_,I_);           % calculated wassersaeule
err = zeros(N_,I_);           % errors of estimated velocities

% temp values that can be overwritten
vv = zeros(I_,1);             % temp v
KK = zeros(I_,1);             % temp K
tmp_x = 0;                    % a result of a query
tmp_i = 0;                    % the index of the result of a query

% definitive values
A = zeros(N_,1);              % cross section's area
p = zeros(N_,1);              % cross section's perimeter
h_t = zeros(N_,1);            % theoretical river height
v_t = zeros(N_,1);            % theoretical flow velocity
hdef = zeros(N_,1);           % good values of wassersaule 
he = zeros(N_,1);             % energy loss
v = zeros(N_,1);              % flow velocities
K = zeros(N_,1);              % channel conveyance factor
sfm = zeros(N_,1);            % middle friction slope
sc = zeros(N_,1);             % river-channel slope
tau = zeros(N_,1);            % shear stress
error = zeros(N_,1);          % section where the algorithm does not converge

% physical constants
g_ = 9.81;                    % gravitational acceleration
rho_ = 1000;                  % density of the fluid
ip_ = [33.3;500;1670];           % incipient motion for some classes of grain size
ip_label_ =  strvcat('granule','pebble','cobble');

% BOUNARY CONDITIONS
Q_ = 160;                     % streamflow
h(1,1) = 5;                   % water elevation at the start point

% calculations for the startpoint
A(1) = get_area(b1(1),b2(1),h1(1),b3(1),h2(1),h(1,1));
p(1) = get_perimeter(b1(1),b2(1),h1(1),b3(1),h2(1),h(1,1));
K(1) = 1 / n_manning(1) * A(1) * (A(1) / p(1) )^(2/3);
v(1) = Q_/A(1);
hdef(1) = h(1,1);

for n=2:N_
    i = 0;
    while i <= I_-1
        i = i + 1;
        
        switch i
            % estimation ot the water depth
            case {1}; h(n,i) = hdef(n-1);
            case {2}; h(n,i) = h(n,i-1) + 0.7*err(n,i-1);
            otherwise;
                err_diff = h(n,i-1) - h_c(n,i-1) + err(n,i-2);
                if (abs(err_diff) >= 0.01);
                    h(n,i) = h(n,i-2) - err(n,i-2)*( h(n,i-2) - h(n,i-1) )/(err_diff);
                else
                    h(n,i) = (h(n,i-1) + h_c(n,i-1)) / 2; 
                end
        end
        
        if( i > 1 )
            % limitation of the changerate
            change_rate = h(n,i) / h(n,i-1);
            if ( change_rate < 0.5 )
                h(n,i) = 0.5 * h(n,i-1);
            elseif ( change_rate > 2)
                h(n,i) = 2 * h(n,i-1);
            end
        end
        
        % limitation of the water elevation below the channel
        if( h(n,i) > h2(n) )
            h(n,i) = h2(n);
        end
        
        % area and wetted perimeter
        A(n) = get_area(b1(n),b2(n),h1(n),b3(n),h2(n),h(n,i));
        p(n) = get_perimeter(b1(n),b2(n),h1(n),b3(n),h2(n),h(n,i));        
        
        % theorical values
        sc(n) = (z(n)-z(n-1))/(l(n-1)-l(n));
        h_t(n) = ( (n_manning(n) * Q_)/(sqrt(sc(n))*b2(n)) )^(3/5);
        v_t(n) = 1 / n_manning(n) * ( (h_t(n)*b2(n))/(b2(n)+2*h_t(n)) )^(2/3)*sqrt(sc(n));
        
        % calculate friction generated by the estimated depth
        KK(i) = A(n) / n_manning(n) * (A(n) / p(n))^(2/3);
        vv(i) = Q_ / A(n);
       
        sfm(n) = (2*Q_ / (K(n-1) + KK(i)))^2;
        he(n) = (l(n-1)-l(n)) * sfm(n);
        
        % real water depth generated with energy balance
        h_c(n,i) = z(n-1) - z(n) + ( v(n-1)^2 - vv(i)^2)/(2*g_) + hdef(n-1) + he(n);
        
        % checkpoint
        err(n,i) = h_c(n,i) - h(n,i);
        sprintf('%i,%i; h=%c err=%c', n, i, h(n,i), err(n,i));
        
        if(abs(err(n,i)) <= ERROR_TOLERANCE_ )
            hdef(n) = h(n,i);
            v(n) = vv(i);
            K(n) = KK(i);           
            break;
        end        
    end
    
    % in the case that the algorithm does not converge
    if( i == I_)
        if ( hdef(n) == 0 )
            [tmp_x, tmp_i] = min( abs(err(n,:)) );
            hdef(n) = h(n, tmp_i);
            v(n) = vv(tmp_i);
            K(n) = KK(tmp_i);
            error(n) = 1;
        else
            sprintf('nothing is impossible..')
            return;
        end
    end
end

% shear stress
tau = rho_.*g_.*hdef.*sc(n);

figure(1);
clf;

subplot(2,1,1);
hold on;
plot(l,hdef,'xr');
plot(l,v,'xb');
plot(l,he,'xm');
xlabel('l [m]'); ylabel('value'); title('backwater curve');
legend('h [m]','v [m.s^{-1}]', 'he [m]');
hold off;
%axis([0,5000,0,2]);

for n=1:N_
    if(changepoint(n) == 1)
        vline(l(n),'-g');
    end
end

subplot(2,1,2);
plot(l,tau,'x');
xlabel('l [m]'); ylabel('\tau [Pa]'); title('shear stess distribution');
%axis([0,5000,0,100]);

for n=1:length(ip_)
    ip_(n);
    hline(ip_(n),'-r',ip_label_(n,:));
end

for n=1:N_
    if(changepoint(n) == 1)
        vline(l(n),'-g');
    end
end
